/*
THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES INC. ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR NON-INFRINGEMENT, ARE
DISCLAIMED. IN NO EVENT SHALL ANALOG DEVICES INC. BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

YOU ASSUME ANY AND ALL RISK FROM THE USE OF THIS CODE OR SUPPORT FILE.

IT IS THE RESPONSIBILITY OF THE PERSON INTEGRATING THIS CODE INTO AN APPLICATION
TO ENSURE THAT THE RESULTING APPLICATION PERFORMS AS REQUIRED AND IS SAFE.

    Module       : uart.c
    Description  : uart interface
    Date         : 10 June 2010
    Version      : v1.00
    Changelog    : v1.00 Initial
*/
#include <include.h>
#include <stdio.h>
#include <string.h>
#include <stddef.h>

#define CR     0x0D

//*****************************************************************************
// Function    : Urt_DataAvail
// Description : Return number of uart data bytes available
//*****************************************************************************
int Urt_DataAvail(void)
{
   // Wait for character available
   if(COMLSR & 0x01)
      return 0x1;
   else
      return 0x0;
}

/*************************************************************************/
/* void Urt_Init(void)                                                   */
/*************************************************************************/
void          Urt_Init        (UrtBaudRates BR)
{
   // Configure the UART
   switch(BR)
      {
      case UART_BAUD_19200:
         // 19200,8,N,1, assming a 16 MHz core clock
         COMDIV = 0x1A;
         COMLCR = COMLCR_WLS_8BITS;
         break;
      case UART_BAUD_115200:
          // 115200,8,N,1, assming a 16 MHz core clock
         COMDIV = 0x4;            // 16M/(16*BR)
         COMLCR = COMLCR_WLS_8BITS | COMLCR_BRK_CLR;
         COMFBR = 0x88AE;        // 115200 based on user guide calcs
         break;
      default:
         break;
      }
}

/*************************************************************************/
/* void Urt_PinSetup(boolean bHandShaking)                                  */
/*************************************************************************/
void Urt_PinSetup(boolean bHandShaking)
{
   if(bHandShaking)
      {
      // Enable the UART functionality on P0.6\P0.7
      GP0CON &= ~(GP0CON_CON6_MSK     | GP0CON_CON7_MSK    );
      GP0CON |=  (GP0CON_CON6_UARTRTS | GP0CON_CON7_UARTCTS);
      }
   // Enable the UART functionality on P1.0\P1.1
   GP1CON &= ~(GP1CON_CON0_MSK     | GP1CON_CON1_MSK    );
   GP1CON |=  (GP1CON_CON0_UART0RXD| GP1CON_CON1_UART0TXD);
}

/*************************************************************************/
/* size_t __read(int handle,unsigned char *buf,size_t bufSize)           */
/*              Write data to a stream                                   */
/*        Needed for retargetting the IAR DLIB library for the ADUCRF101  */
/*************************************************************************/
size_t __read(int handle,unsigned char *buf,size_t bufSize)
{
   size_t i;
   for (i=0x0; i<bufSize;i++)
      {
      // Wait for character available
      while(!(COMLSR_DR==(COMLSR & COMLSR_DR)));
      buf[i] = COMRX;
      }
   return i;
}

/*************************************************************************/
/* __write(int handle,const unsigned char *buf,size_t bufSize)           */
/*              Read data from a stream                                  */
/*        Needed for retargetting the IAR DLIB library for the ADUCRF101  */
/*************************************************************************/
size_t __write(int handle,const unsigned char *buf,size_t bufSize)
{
   size_t i;
   for (i=0x0; i<bufSize;i++)
      {
      if (buf[i] == '\n')
         Urt_DataBytWr(CR);/* output CR */
      Urt_DataBytWr(buf[i]);
      }
   return i;
}


//*****************************************************************************
// Function    : UART_Int_Handler      (NOT USED YET!!)
// Description : UART interrupt handler
//*****************************************************************************
void UART_Int_Handler(void)
{
   if(COMIIR & COMIIR_STA_TXBUFEMPTY)
      {
      // Transmit buffer empty interrupt
      COMTX = '?';
      }

   if(COMIIR & COMIIR_STA_RXBUFFULL)
      {
      (void)COMRX; // Discard the byte
      }

}

//*****************************************************************************
// Function    : Urt_DataBytRd
// Description : Read single byte from uart (blocking)
//*****************************************************************************
unsigned char Urt_DataBytRd(void)
{
   // Wait for character available
   while(!(0x01==(COMLSR & 0x01)));
   return (COMRX);
}


//*****************************************************************************
// Function    : Urt_DataBytWr
// Description : Write single byte to uart (blocking)
//*****************************************************************************
void Urt_DataBytWr(unsigned char ucByte)
{
   while(!(0x020==(COMLSR & 0x020)));
   COMTX = ucByte;
}





